const DATE_TIME_FORMAT = "YYYY-MM-DD HH:mm:ss"

function joinTimestamp(join, restful = false) {
  if (!join) return restful ? "" : {}

  const now = new Date().getTime()
  if (restful) return `?_t=${now}`

  return { _t: now }
}

/**
 * @description: Format request parameter time
 */
function formatRequestDate(params) {
  if (Object.prototype.toString.call(params) !== "[object Object]") return

  for (const key in params) {
    const format = params[key]?.format ?? null
    if (format && typeof format === "function")
      params[key] = params[key].format(DATE_TIME_FORMAT)

    if (isString(key)) {
      const value = params[key]
      if (value) {
        try {
          params[key] = isString(value) ? value.trim() : value
        } catch (error) {
          throw new Error(error)
        }
      }
    }
    if (isObject(params[key])) formatRequestDate(params[key])
  }
}


function setObjToUrlParams(baseUrl, obj) {
  let parameters = ""
  for (const key in obj) parameters += `${key}=${encodeURIComponent(obj[key])}&`

  parameters = parameters.replace(/&$/, "")
  return /\?$/.test(baseUrl)
    ? baseUrl + parameters
    : baseUrl.replace(/\/?$/, "?") + parameters
}


function doRefreshToken(params) {
  return defHttp.post(
    {
      url: '/auth/token',
      data: params,
    },
    { joinPrefix: false },
  )
}


class RequestQueue {
  constructor() {
    this.waitingQueue = []
  }

  wait() {
    return new Promise(resolve => [this.waitingQueue.push(resolve)])
  }

  clear() {
    this.waitingQueue.forEach((fn, index) => fn(index))
    this.waitingQueue.length = 0
  }
}

const requestQueue = new RequestQueue()

function isRefreshTokenRequest(url, apiUrl) {
  return url?.replace(apiUrl || "", "") === "/auth/token"
}

function checkIsTokenExp(response) {
  const apiUrl = response?.config?.requestOptions?.apiUrl

  return response.status === 401 && !isRefreshTokenRequest(response.config.url, apiUrl) && !!getJWTToken()
}

let __global__refreshTokenPromise

function doRefreshTokenFromSingleEntry(params ) {
  if (__global__refreshTokenPromise)
    return __global__refreshTokenPromise

  __global__refreshTokenPromise = doRefreshToken(params)

  __global__refreshTokenPromise.finally(() => __global__refreshTokenPromise = null)

  return __global__refreshTokenPromise
}


async function doTokenExpHandler(response, axiosInstance) {
  if (isEmpty(getRefreshToken())) return
  const { token, refreshToken } = await doRefreshTokenFromSingleEntry({
    refreshToken: getRefreshToken()
  })

  setAuthCache(JWT_TOKEN_KEY, token)
  setAuthCache(REFRESH_TOKEN_KEY, refreshToken)

  const result = await axiosInstance(response.config)
  requestQueue.clear()
  return result
}

/**
 * @description: 数据处理，方便区分多种处理方式
 */
const transform = {
  /**
   * @description: 处理响应数据。如果数据不是预期格式，可直接抛出错误
   */
  transformResponseHook: (res, options) => {
    const { isReturnNativeResponse } = options
    // 是否返回原生响应头 比如：需要获取响应头时使用该属性
    if (isReturnNativeResponse) return res

    return res.data
  },

  // 请求之前处理config
  beforeRequestHook: (config, options) => {
    const {
      apiUrl,
      joinPrefix,
      joinParamsToUrl,
      formatDate,
      joinTime = true,
      urlPrefix = ""
    } = options

    if (joinPrefix) config.url = `${urlPrefix}${config.url}`
    if (apiUrl && isString(apiUrl)) config.url = `${apiUrl}${config.url}`

    const params = config.params || {}
    const data = config.data || false
    formatDate && data && !isString(data) && formatRequestDate(data)
    if (config.method?.toUpperCase() === RequestEnum.GET) {
      if (!isString(params)) {
        // 给 get 请求加上时间戳参数，避免从缓存中拿数据。
        config.params = Object.assign(
          params || {},
          joinTimestamp(joinTime, false)
        )
      } else {
        // 兼容restful风格
        config.url = `${config.url + params}${joinTimestamp(joinTime, true)}`
        config.params = undefined
      }
    } else {
      if (!isString(params)) {
        formatDate && formatRequestDate(params)
        if (
          Reflect.has(config, "data") &&
          config.data &&
          (Object.keys(config.data).length > 0 ||
            config.data instanceof FormData)
        ) {
          config.data = data
          config.params = params
        } else {
          // 非GET请求如果没有提供data，则将params视为data
          config.data = params
          config.params = undefined
        }
        if (joinParamsToUrl) {
          config.url = setObjToUrlParams(
            config.url,
            Object.assign({}, config.params, config.data)
          )
        }
      } else {
        // 兼容restful风格
        config.url = config.url + params
        config.params = undefined
      }
    }
    return config
  },

  /**
   * @description: 请求拦截器处理
   */
  requestInterceptors: async (config, options) => {
    // 请求之前处理config
    const token = getJWTToken()

    if (token && config?.requestOptions?.withToken !== false) {
      // try {
      //   const { exp = 0 } = jwtDecode(token)

      //   if ((exp * 1000 - 1000 * 30) <= Date.now()) {
      //     if (!isRefreshTokenRequest(config.url)) {
      //       await requestQueue.wait()
      //     }
      //   }
      // }
      // catch (e) {
      // }

      config.headers.Authorization = options.authenticationScheme
        ? `${options.authenticationScheme} ${token}`
        : token
    }
    return config
  },

  /**
   * @description: 响应拦截器处理
   */
  responseInterceptors: res => {
    return res
  },

  /**
   * @description: 响应错误处理
   */
  responseInterceptorsCatch: async (axiosInstance, error) => {
    const { response, code, message, config } = error || {}
    const msg =
      response?.data?.error?.message ||
      response?.data?.message ||
      response?.data?.msg ||
      response?.data
    const err = error?.toString?.() ?? ""
    let errMessage = ""

    if (axios.isCancel(error)) return Promise.reject(error)

    try {
      if (checkIsTokenExp(response))
        return await doTokenExpHandler(response, axiosInstance)

      if (code === "ECONNABORTED" && message.includes("timeout"))
        errMessage = RED._("api.apiTimeoutMessage")

      if (err?.includes("Network Error"))
        errMessage = RED._("api.networkExceptionMsg")

      if (errMessage) {
        RED.notifications.notify(errMessage, 'error')

        return Promise.reject(error)
      }
    } catch (error) {
      throw new Error(error)
    }

    checkStatus(error?.response?.status, msg)

    return Promise.reject(error)
  }
}

function createAxios(opt) {
  return new VAxios( // 深度合并
    _.mergeWith(
      {
        // See https://developer.mozilla.org/en-US/docs/Web/HTTP/Authentication#authentication_schemes
        authenticationScheme: "Bearer",
        timeout: 10 * 1000,
        // 基础接口地址
        // baseURL: apiUrl,

        headers: { "Content-Type": ContentTypeEnum.JSON },
        // 数据处理方式
        transform: _.clone(transform),
        // 配置项，下面的选项都可以在独立的接口请求中覆盖
        requestOptions: {
          // 默认将prefix 添加到url
          joinPrefix: true,
          // 是否返回原生响应头 比如：需要获取响应头时使用该属性
          isReturnNativeResponse: false,
          // 需要对返回数据进行处理
          isTransformResponse: true,
          // post请求的时候添加参数到url
          joinParamsToUrl: false,
          // 格式化提交参数时间
          formatDate: true,
          // 接口地址
          apiUrl: '/api',
          // 接口拼接地址
          urlPrefix: '/yt',
          //  是否加入时间戳
          joinTime: true,
          // 忽略重复请求
          ignoreCancelToken: true,
          // 是否携带token
          withToken: true,
          retryRequest: {
            isOpenRetry: false,
            count: 5,
            waitTime: 100
          }
        }
      },
      opt || {}
    )
  )
}

const defHttp = createAxios()
