const { app, BrowserWindow, Menu, ipcMain, dialog, protocol, session, globalShortcut } = require('electron');
const path = require('path');
const { Writable } = require('stream');
const fs = require('fs').promises;
const fsSync = require('fs');
const { spawn, exec, execFile } = require('child_process');
const sudoPrompt = require('sudo-prompt');
const os = require('os');
const Store = require('electron-store');
const psTree = require('ps-tree');
const {SerialPort}=require('serialport')
app.commandLine.appendSwitch('ignore-certificate-errors');
app.commandLine.appendSwitch('allow-insecure-localhost');
// 配置和状态管理
const configStore = new Store();
const state = {
  uiWindow: null,
  nodeRedProcess: null,
  mqttProcess: null,
  currentToken: null,
  currentAction: null,
  currentCallback: null,
  tagType: null
};

// 确保单实例运行
const gotTheLock = app.requestSingleInstanceLock();
if (!gotTheLock) {
  app.quit();
} else {
  app.on('second-instance', (event, commandLine, workingDirectory) => {
    // 当尝试运行第二个实例时，聚焦到主窗口
    if (state.uiWindow) {
      if (state.uiWindow.isMinimized()) state.uiWindow.restore();
      state.uiWindow.focus();
    }
  });
}

// 禁用默认菜单
Menu.setApplicationMenu(null);



// 监听渲染进程的请求，返回串口列表
ipcMain.handle('get-serial-ports', async () => {
  try {
    // 获取所有可用的串口
    const ports = await SerialPort.list();
    // 提取端口信息，只返回需要的字段
    return ports.map(port => ({
      comName: port.path,       // 端口号，如 COM3 或 /dev/ttyUSB0
      manufacturer: port.manufacturer,  // 制造商
      vendorId: port.vendorId,  // 供应商ID
      productId: port.productId  // 产品ID
    }));
  } catch (error) {
    console.error('获取串口列表失败:', error);
    throw error;
  }
});

// 创建主窗口
const createWindow = () => {
  if (state.uiWindow) return; // 防止重复创建窗口
  
  state.uiWindow = new BrowserWindow({
    width: 1400,
    height: 900,
    title: '物联网测试验证平台',
    autoHideMenuBar: true,
    webPreferences: {
      webSecurity: false,
      webviewTag: true,
      nodeIntegrationInSubFrames: true,
      nodeIntegration: true,
      contextIsolation: true,
      preload: path.join(__dirname, 'preload.js'),
      session: {
        cache: false
      }
    }
  });
  
  state.uiWindow.on('close', (event) => {
    // 在关闭窗口前优雅地停止子进程
    stopServices();
    // 彻底关闭应用
    app.quit();
  });


  state.uiWindow.webContents.openDevTools()
  state.uiWindow.loadFile(path.join(__dirname, 'web/index.html'));

};


// 注册全局快捷键并打开指定页面
function registerNodredShortcut() {
  const ret = globalShortcut.register('Control+Shift+N', () => {
    console.log('Ctrl+Shift+N pressed');

    // 创建新窗口或导航到指定页面
    const newWindow = new BrowserWindow({
      width: 1400,
      height: 900,
    });
    
    // 打开你的指定页面（例如：special.html）
    //newWindow.loadFile('special.html');
    // 或者使用URL：
    newWindow.loadURL('http://127.0.0.1:1955');
  });

  if (!ret) {
    console.error('Shortcut registration failed');
  }
}



// 用户数据目录和配置文件管理
const userDataPath = app.getPath('userData');
const configPath = path.join(userDataPath, 'serverConfig.json');

// 初始化配置文件 
async function initConfigFile() {
  try {
    await fs.access(configPath);
  } catch {
    await fs.writeFile(configPath, JSON.stringify({ serverUrl: '' }, null, 2));
  }
}

// 读取服务地址
async function getServerUrl() {
  await initConfigFile();
  const content = await fs.readFile(configPath, 'utf8');
  return JSON.parse(content).serverUrl || '';
}

// 写入服务地址
async function setServerUrl(url) {
  await initConfigFile();
  const content = await fs.readFile(configPath, 'utf8');
  const config = JSON.parse(content);
  config.serverUrl = url;
  await fs.writeFile(configPath, JSON.stringify(config, null, 2));
}

// IPC 处理函数
function handleSetTitle(event, title) {
  const win = BrowserWindow.fromWebContents(event.sender);
  win.setTitle(title);
  win.webContents.executeJavaScript('result("调用成功")');
}

function handleAction(event, action, callback) {
  const json = JSON.parse(action);
  console.log('TAG:handleAction ', 'action=', json.action, ';callback=', callback, ';tagType:', json.tagType);
  
  state.tagType = json.tagType;
  state.currentAction = json.action;
  state.currentCallback = callback;
  
  switch (json.action) {
    case 'OPEN_OCR':
      // createTakePhotoOCR()
      break;
    default:
      console.log('未知操作:', json.action);
  }
}

function setToken(event, token) {
  console.log("setToken:", token);
  state.currentToken = token;
}

function getToken() {
  console.log("getToken:", state.currentToken);
  return state.currentToken;
}

let node_path = `${__dirname}/node/v18.18.0/bin/node`

// 服务管理
async function startNodeRed() {
  let server_url = await getServerUrl();
  updateHostIP(`${__dirname}/nr/node-red/packages/node_modules/@node-red/editor-client/public/red/red.min.js`,server_url);
  const targetDir = path.join(__dirname, 'nr');
  const nodeRedUserDir = path.join(userDataPath, 'nodered-data'); // 用户专属目录
 
  // 确保目录存在并设置权限
  fs.mkdir(nodeRedUserDir, { recursive: true }).then(() => {
    fs.chmod(nodeRedUserDir, 0o755).catch(console.error); // 设置目录权限
 
    // 启动Node-RED时指定用户目录
    state.nodeRedProcess = spawn(node_path, [
      'main.js',
      '-c=config.js',
      `--userDir=${nodeRedUserDir}` // 关键修改
    ], {
      cwd: targetDir,
      stdio: 'inherit',
      env: {
        ...process.env,
        NODE_ENV: process.env.NODE_ENV || 'production'
      }
    });
 
    // 保持原有事件监听
    state.nodeRedProcess.on('close', (code) => {
      console.log(`Node-RED 进程已退出，退出码 ${code}`);
    });
    
    state.nodeRedProcess.on('error', (err) => {
      console.error('启动 Node-RED 时发生错误:', err);
    });
  }).catch(console.error);
}

/**
 * 简单替换文件中的 HOST_IP 常量值
 * @param {string} filePath - 要修改的文件路径
 * @param {string} newIP - 新的 IP 地址
 */
function updateHostIP(filePath, newIP) {
  try {
    console.log(filePath);
    // 读取文件内容
    let content =  fsSync.readFileSync(filePath, 'utf8');
    
    // 使用简单正则匹配并替换
    const updatedContent = content.replace(
      /(const\s+HOST_IP\s*=\s*['"]?)[^'";]*(['"]?\s*;)/, 
      `$1${newIP}$2`
    );
    
    // 检查是否实际修改了内容
    if (updatedContent === content) {
      throw new Error('未找到 HOST_IP 声明');
    }
    
    // 写回文件
    fsSync.writeFileSync(filePath, updatedContent, 'utf8');
    console.log(`HOST_IP 已更新为: ${newIP}`);
    return true;
  } catch (error) {
    console.error('修改失败:', error.message);
    return false;
  }
}

function startMQTT() {
  const mqttTargetDir = path.join(__dirname, 'mqtt');
  const mqttUserDir = path.join(userDataPath, 'mqtt-data'); // 用户专属目录
 
  fs.mkdir(mqttUserDir, { recursive: true }).then(() => {
    fs.chmod(mqttUserDir, 0o755).catch(console.error);
 
    state.mqttProcess = spawn(node_path, ['mqtt.js'], {
      cwd: mqttTargetDir,
      stdio: 'inherit',
      env: {
        ...process.env,
        DATA_DIR: mqttUserDir // 传递自定义环境变量
      }
    });
 
    // 保持原有事件监听
    state.mqttProcess.on('close', (code) => {
      console.log(`MQTT 进程已退出，退出码 ${code}`);
    });
    
    state.mqttProcess.on('error', (err) => {
      console.error('启动 MQTT 服务时发生错误:', err);
    });
  }).catch(console.error);
}
/**
 * 清除占用指定端口的进程
 * @param {number} port - 端口号（如 10883）
 * @returns {Promise<void>}
 */
function clearPortProcess(port) {
  return new Promise((resolve, reject) => {
    // 根据系统选择命令（Linux/macOS 使用 lsof，Windows 使用 netstat）
    let command;
    if (process.platform === 'win32') {
      // Windows：查找端口对应的PID并终止
      command = `netstat -ano | findstr :${port} | findstr LISTENING | for /f "tokens=5" %a in ('more') do taskkill /F /PID %a`;
    } else {
      // Linux/macOS：查找端口对应的PID并终止
      command = `lsof -t -i :${port} | xargs -r kill -9`;
    }

    exec(command, (error, stdout, stderr) => {
      if (error) {
        // 忽略"未找到进程"的错误（非致命）
        if (stderr.includes('找不到') || stdout.trim() === '') {
          console.log(`端口 ${port} 未被占用，无需清理`);
          return resolve();
        }
        return reject(new Error(`清理端口失败: ${stderr}`));
      }
      console.log(`已成功终止占用端口 ${port} 的进程`);
      resolve();
    });
  });
}


function stopServices() {
  // 停止 Node-RED 进程
  if (state.nodeRedProcess) {
    clearPortProcess(1955)
    clearPortProcess(3002)
    state.nodeRedProcess.kill('SIGTERM');
    state.nodeRedProcess = null;
  }
  
  // 停止 MQTT 进程
  if (state.mqttProcess) {
    clearPortProcess(10883)
    state.mqttProcess.kill('SIGTERM');
    state.mqttProcess = null;
  }
}

// 应用生命周期管理
app.whenReady().then(async () => {
  await initConfigFile();
  
  // 注册IPC事件
  ipcMain.on('set-title', handleSetTitle);
  ipcMain.on('doAction', handleAction);
  ipcMain.handle('set-token', setToken);
  ipcMain.handle('get-token', getToken);
  ipcMain.handle('get-server-url', getServerUrl);
  ipcMain.handle('set-server-url', (_, url) => setServerUrl(url));

  // 注册自定义协议
  protocol.registerFileProtocol('atom', (request, callback) => {
    const url = request.url.substr(7);
    const dURL = decodeURI(path.normalize(url));
    callback(dURL);
  });

  // 处理命令行参数
  global.sharedObject = { argv: process.argv };
  console.log("sharedObject:", global.sharedObject.argv);
  const token1 = global.sharedObject.argv[1];
  const token2 = global.sharedObject.argv[2];
  console.log("token1:", token1, ",token2:", token2);

  // 创建窗口
  createWindow();
  
  // 启动服务
  startNodeRed();
  startMQTT();

  // 注册全局快捷键（开发模式）
  if (process.env.NODE_ENV === 'development') {
    globalShortcut.register('CommandOrControl+R', () => {
      if (state.uiWindow) state.uiWindow.reload();
    });
    
    globalShortcut.register('CommandOrControl+Alt+I', () => {
      if (state.uiWindow) state.uiWindow.webContents.toggleDevTools();
    });
  }

  registerNodredShortcut();


});

// 应用退出时清理资源
app.on('will-quit', () => {
  stopServices();
  // 取消注册所有快捷键
  globalShortcut.unregisterAll();
});

// 窗口全部关闭时的处理
app.on('window-all-closed', () => {
  if (process.platform !== 'darwin') {
    app.quit();
  }
});

// 应用激活时的处理
app.on('activate', () => {
  if (BrowserWindow.getAllWindows().length === 0) {
    createWindow();
  }
});    
